﻿using NVCC.Models;
using NVCC.WebUI.Infrastructure;
using NVCC.WebUI.Models;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web.Mvc;

namespace NVCC.WebUI.Controllers
{
    public class UserController : Controller
    {
        private readonly IUserService _userService;

        public UserController(IUserService userService)
        {
            _userService = userService;
        }

        // GET: User
        public ActionResult _UserInfo()
        {
            return PartialView(new UserInfoViewModel { User = _userService.GetUser() });
        }

        public ActionResult _ReportMenu()
        {
            ViewBag.Title = TempData["Title"] as string;
            User user = _userService.GetUser();
            if (user != null && user.CanViewReports)
            {
                return PartialView();
            }
            else
            {
                return new EmptyResult();
            }
        }

        public ActionResult UserInfo()
        {
            return View(new UserInfoViewModel { User = _userService.GetUser() });
        }

        public ActionResult Login(string returnController, string returnAction)
        {
            ViewBag.ErrorMessage = TempData["ErrorMessage"] as string;
            ViewBag.InfoMessage = TempData["InfoMessage"] as string;
            User user = _userService.GetUser();
            if (user == null)
            {
                return RedirectToAction("Unauthorized", "Home");
            }
            short defaultSta3n = user.CurrentDefaultFacility;
            // if no current facility, use last logged into
            if (defaultSta3n == 0)
            {
                UserFacility uf = user.Facilities.Values.OrderByDescending(f => f.LastAuthDateTime).FirstOrDefault();
                if (uf != null && uf.LastAuthDateTime != null)
                {
                    defaultSta3n = uf.sta3n;
                }
            }
            // if still none, use first for which user is admin
            if (defaultSta3n == 0)
            {
                UserFacility uf = user.Facilities.Values.Where(f => f.Admin).FirstOrDefault();
                if (uf != null)
                {
                    defaultSta3n = uf.sta3n;
                }
            }
            IEnumerable<UserFacility> availableFacilities = user.Facilities.Values.Where(f => (f.PHIPII ?? false));
            if (availableFacilities.Count() == 0)
            {
                return RedirectToAction("PHIPII", "Help");
            }
            return View(new LoginViewModel { AvailableFacilities = availableFacilities, sta3n = defaultSta3n, ReturnController = returnController, ReturnAction = returnAction });
        }

        [HttpPost]
        [ValidateAntiForgeryToken]
        public ActionResult Login([Bind(Include = "UserName, Password, sta3n, ReturnAction, ReturnController")] LoginViewModel login)
        {
            User user = _userService.GetUser();
            if (user == null)
            {
                return RedirectToAction("Unauthorized", "Home");
            }
            // Fill in the rest of the view model in case there is an error and it gets sent to the View.
            login.AvailableFacilities = user.Facilities.Values.Where(f => (f.PHIPII ?? false));
            if (login.sta3n == 0 || login.UserName == null || login.Password == null)
            {
                @ViewBag.ErrorMessage = "You must select a station and provide Access and Verify codes!";
                return View(login);
            }
            if (!user.Facilities.ContainsKey(login.sta3n))
            {
                ModelState.AddModelError("sta3n", "Must select a known station");
                return View(login);
            }
            if (!user.Facilities[login.sta3n].PHIPII ?? false)
            {
                ModelState.AddModelError("sta3n", "Must select a station for which you have PHI/PII");
                return View(login);
            }

            ViaUser vistaUser = _userService.GetViaUser(login.UserName, login.Password, user.Facilities[login.sta3n].sta6a, login.sta3n);

            if (vistaUser.FaultMessage != null && vistaUser.FaultMessage.Contains("Division")
                && vistaUser.FaultMessage.Contains("not supported"))
            {
               return RedirectToAction("Division", "User", new { station = login.sta3n });
            }

            if (vistaUser.InvalidAccessVerifyCode )
            {
                if (vistaUser.FaultMessage!=null && vistaUser.FaultMessage.Contains("VERIFY CODE must be changed"))
                    @ViewBag.ErrorMessage = "Your VERIFY CODE must be changed in VistA/CPRS before continued use.";
                else
                    @ViewBag.ErrorMessage = "Your Access and/or Verify code is not correct. Please try again.";
                return View(login);
            }

            if (vistaUser.FaultMessage != null && vistaUser.ViaBApproved == true)
            { 
                TempData["Message"] = vistaUser.FaultMessage;
                return RedirectToAction("Error", "Home");
            }
            user.Facilities[login.sta3n].LastAuthDateTime = DateTime.UtcNow;
            user.Facilities[login.sta3n].ViaB = vistaUser.ViaBApproved;
            user.Facilities[login.sta3n].VIAToken = vistaUser.QueryBean.ViaProvider.UserSessionId;
            user.Facilities[login.sta3n].VIATokenExpiration = DateTime.UtcNow.AddHours(4);

            if (vistaUser.Divisions.Count() > 0 && !vistaUser.Divisions.Any(d => d.Sta6a == user.Facilities[login.sta3n].sta6a))
            {
                // the current sta6a is not one of the divisions returned in the ViaUser
                user.Facilities[login.sta3n].sta6a = vistaUser.Divisions.First().Sta6a;
                user.Facilities[login.sta3n].LocationName = vistaUser.Divisions.First().DivisionName;
            }

            user.CurrentDefaultFacility = login.sta3n;

            _userService.UpdateUserFacility(user);

            //Store encoded credentials in a session variable for imaging service basic Auth requests
            string encodedCredentials = Convert.ToBase64String(ASCIIEncoding.ASCII.GetBytes(login.UserName + ":" + login.Password));
            HttpContextManager.Current.Session["ImageAuthToken"] = encodedCredentials;

            if (string.IsNullOrWhiteSpace(login.ReturnController) || string.IsNullOrWhiteSpace(login.ReturnAction))
            {
                return RedirectToAction("Index", "Home");
            }
            else
            {
                return RedirectToAction(login.ReturnAction, login.ReturnController);
            }
        }

        [NvccAuthorize]
        public ActionResult StationInfo()
        {
            User user = _userService.GetUser();
            short station = user.CurrentDefaultFacility;

            var stationInfo = _userService.GetStationInfo(station);

            var stationInfoViewModel = new StationInfoViewModel
            {
                MedCtr = stationInfo.MedCtr,
                FullAddress = stationInfo.FullAddress,
                FacilityPhoneNumber = stationInfo.FacilityPhoneNumber,
                FacilityFaxNumber = stationInfo.FacilityFaxNumber,
                sta3n = station,
                Admin = user.Facilities[station].Admin
            };

            return View(stationInfoViewModel);
        }

        [HttpPost]
        [ValidateAntiForgeryToken]
        [NvccAuthorize]
        public ActionResult StationInfo([Bind(Include = "MedCtr, FullAddress, FacilityPhoneNumber, FacilityFaxNumber, sta3n")] StationInfoViewModel stationInfoViewModel)
        {
            User user = _userService.GetUser();
            // Don't trust stationInfoViewModel.sta3n
            if (!user.Facilities.ContainsKey(stationInfoViewModel.sta3n))
            {
                return RedirectToAction("Unauthorized", "Home");
            }
            UserFacility userFacility = user.Facilities[stationInfoViewModel.sta3n];
            if (!userFacility.Admin)
            {
                return RedirectToAction("Unauthorized", "Home");
            }

            var stationInfo = _userService.GetStationInfo(stationInfoViewModel.sta3n);
            stationInfo.MedCtr = stationInfoViewModel.MedCtr;
            stationInfo.FullAddress = stationInfoViewModel.FullAddress;
            stationInfo.FacilityPhoneNumber = stationInfoViewModel.FacilityPhoneNumber;
            stationInfo.FacilityFaxNumber = stationInfoViewModel.FacilityFaxNumber;

            _userService.SaveStationInfo(stationInfo);

            return RedirectToAction("Index", "Home");
        }

        [NvccAuthorize]
        public ActionResult Division(short? station)
        {
            ViewBag.InfoMessage = TempData["InfoMessage"];
            User user = _userService.GetUser();
            short sta3n = station ?? user.CurrentDefaultFacility;
            if (!user.Facilities.ContainsKey(sta3n))
            {
                return RedirectToAction("Unauthorized", "Home");
            }
            ViaUser viaUser = _userService.GetViaUser(sta3n);
            DivisionViewModel divisionViewModel = new DivisionViewModel();

            if ((viaUser != null) && (viaUser.Divisions.Count() > 0))
            {
                divisionViewModel.Divisions = viaUser.Divisions;
            }
            else
            {
                divisionViewModel.Divisions = _userService.GetDivisions(sta3n);
            }
            divisionViewModel.sta6a = user.Facilities[sta3n].sta6a;
            divisionViewModel.sta3n = sta3n;
            user.CurrentDefaultFacility = sta3n;
            return View(divisionViewModel);
        }

        [HttpPost]
        [ValidateAntiForgeryToken]
        [NvccAuthorize]
        public ActionResult Division([Bind(Include = "sta6a, sta3n")] DivisionViewModel divisionViewModel)
        {
            User user = _userService.GetUser();
            if (!user.Facilities.ContainsKey(divisionViewModel.sta3n))
            {
                return RedirectToAction("Unauthorized", "Home");
            }
            ViaUser viaUser = _userService.GetViaUser(divisionViewModel.sta3n);

            if ((viaUser != null) && (viaUser.Divisions.Count() > 0))
            {
                divisionViewModel.Divisions = viaUser.Divisions;
            }
            else
            {
                divisionViewModel.Divisions = _userService.GetDivisions(divisionViewModel.sta3n);
            }
            user.CurrentDefaultFacility = divisionViewModel.sta3n;

            if (string.IsNullOrEmpty(divisionViewModel.sta6a))
            {
                ModelState.AddModelError("sta6a", "You must choose a station location.");
                return View(divisionViewModel);
            }
            if (!divisionViewModel.Divisions.Any(d => d.Sta6a == divisionViewModel.sta6a))
            {
                ModelState.AddModelError("sta6a", "Selected value not in list.");
                return View(divisionViewModel);
            }

            user.Facilities[divisionViewModel.sta3n].sta6a = divisionViewModel.sta6a;
            user.Facilities[divisionViewModel.sta3n].LocationName =
                divisionViewModel.Divisions.Where(d => d.Sta6a == divisionViewModel.sta6a).First().DivisionName;
            _userService.UpdateUserFacility(user);

            return RedirectToAction("Index", "Home");
        }
    }
}